<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

// Add admin menu for theme settings
function mytheme_add_admin_menu() {
    add_menu_page(
        'Theme Settings',           // Page title
        'Hi Classic Settings',      // Menu title
        'manage_options',           // Capability required to access the menu
        'mytheme-settings',         // Menu slug
        'mytheme_settings_page',    // Function to display settings page content
        'dashicons-heading'         // Icon for the menu
    );
}
add_action('admin_menu', 'mytheme_add_admin_menu');

// Register theme settings for header/footer templates, custom header code, custom HTML code, editor choice
function mytheme_register_settings() {
    // Register new settings
    register_setting('mytheme_settings_group', 'mytheme_header_template');
    register_setting('mytheme_settings_group', 'mytheme_footer_template');
    register_setting('mytheme_settings_group', 'mytheme_custom_header_code');  // Custom header code setting
    register_setting('mytheme_settings_group', 'mytheme_custom_html_code');     // Custom HTML code setting
    register_setting('mytheme_settings_group', 'mytheme_editor_choice');        // Editor choice setting

    // Add new sections for Elementor settings
    add_settings_section(
        'mytheme_elementor_section',          // Section ID
        'Elementor Header & Footer',          // Section Title
        'mytheme_elementor_section_callback', // Callback function for section description
        'mytheme-settings'                    // Settings page slug
    );

    add_settings_field(
        'mytheme_header_template',            // Field ID
        'Select Header Template',             // Field title
        'mytheme_header_template_callback',   // Callback function to display field
        'mytheme-settings',                   // Settings page slug
        'mytheme_elementor_section'           // Section ID
    );

    add_settings_field(
        'mytheme_footer_template',            // Field ID
        'Select Footer Template',             // Field title
        'mytheme_footer_template_callback',   // Callback function to display field
        'mytheme-settings',                   // Settings page slug
        'mytheme_elementor_section'           // Section ID
    );

    // Add field for custom header code (JavaScript only)
    add_settings_field(
        'mytheme_custom_header_code',         // Field ID
        'Custom JS Code in Header',           // Field title
        'mytheme_custom_header_code_callback',// Callback function to display field
        'mytheme-settings',                   // Settings page slug
        'mytheme_elementor_section'           // Section ID
    );

    // Add new field for custom HTML code in the header
    add_settings_field(
        'mytheme_custom_html_code',           // Field ID
        'Custom HTML / CSS Code in Header',   // Field title
        'mytheme_custom_html_code_callback',  // Callback function to display field
        'mytheme-settings',                   // Settings page slug
        'mytheme_elementor_section'           // Section ID
    );

    // Add new section for editor settings
    add_settings_section(
        'mytheme_editor_section',             // Section ID
        'Editor Settings',                    // Section Title
        'mytheme_editor_section_callback',    // Callback function for section description
        'mytheme-settings'                    // Settings page slug
    );

    add_settings_field(
        'mytheme_editor_choice',              // Field ID
        'Select Default Editor',              // Field title
        'mytheme_editor_choice_callback',     // Callback function to display field
        'mytheme-settings',                   // Settings page slug
        'mytheme_editor_section'              // Section ID
    );
}
add_action('admin_init', 'mytheme_register_settings');

// Settings page content
function mytheme_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Theme Settings', 'hi-classic'); ?></h1>

        <?php settings_errors(); ?>

        <form method="post" action="options.php">
            <?php
            settings_fields('mytheme_settings_group');
            do_settings_sections('mytheme-settings');
            wp_nonce_field('mytheme_settings_group-options', 'mytheme_nonce_field');  // Adds nonce for security
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Section description for Elementor settings
function mytheme_elementor_section_callback() {
    echo 'Choose your custom Elementor header and footer templates from the saved templates section.';
}

// Display header template field
function mytheme_header_template_callback() {
    $elementor_templates = get_posts(array(
        'post_type' => 'elementor_library',
        'posts_per_page' => -1,
        'post_status' => 'publish',
        'meta_query' => array(
            array(
                'key' => '_elementor_template_type',
                'value' => 'section',
                'compare' => '='
            ),
        ),
    ));

    $header_template = get_option('mytheme_header_template');
    ?>
    <select name="mytheme_header_template" id="mytheme_header_template">
        <option value=""><?php esc_html_e('Default Header', 'hi-classic'); ?></option>
        <?php foreach ($elementor_templates as $template) : ?>
            <option value="<?php echo $template->ID; ?>" <?php selected($header_template, $template->ID); ?>>
                <?php echo esc_html($template->post_title); ?>
            </option>
        <?php endforeach; ?>
    </select>
    <?php
}

// Display footer template field
function mytheme_footer_template_callback() {
    $elementor_templates = get_posts(array(
        'post_type' => 'elementor_library',
        'posts_per_page' => -1,
        'post_status' => 'publish',
        'meta_query' => array(
            array(
                'key' => '_elementor_template_type',
                'value' => 'section',
                'compare' => '='
            ),
        ),
    ));

    $footer_template = get_option('mytheme_footer_template');
    ?>
    <select name="mytheme_footer_template" id="mytheme_footer_template">
        <option value=""><?php esc_html_e('Default Footer', 'hi-classic'); ?></option>
        <?php foreach ($elementor_templates as $template) : ?>
            <option value="<?php echo $template->ID; ?>" <?php selected($footer_template, $template->ID); ?>>
                <?php echo esc_html($template->post_title); ?>
            </option>
        <?php endforeach; ?>
    </select>
    <?php
}

// Display custom JavaScript header code field
function mytheme_custom_header_code_callback() {
    $custom_header_code = get_option('mytheme_custom_header_code');
    ?>
    <textarea name="mytheme_custom_header_code" id="mytheme_custom_header_code" rows="5" cols="50"><?php echo esc_textarea($custom_header_code); ?></textarea>
    <p class="description"><?php esc_html_e('Insert any custom JavaScript code here. It will be included in the <head> section of your website.', 'hi-classic'); ?></p>
    <?php
}

// Display custom HTML code field for header
function mytheme_custom_html_code_callback() {
    $custom_html_code = get_option('mytheme_custom_html_code');
    ?>
    <textarea name="mytheme_custom_html_code" id="mytheme_custom_html_code" rows="5" cols="50"><?php echo esc_textarea($custom_html_code); ?></textarea>
    <p class="description"><?php esc_html_e('Insert any custom HTML / CSS code here. It will be included in the <head> section of your website.', 'hi-classic'); ?></p>
    <?php
}

// Editor section description
function mytheme_editor_section_callback() {
    echo 'Choose your default editor: (Classic Editor or Gutenberg.)';
}

// Display editor choice field
function mytheme_editor_choice_callback() {
    $editor_choice = get_option('mytheme_editor_choice', 'classic');
    ?>
    <select name="mytheme_editor_choice" id="mytheme_editor_choice">
        <option value="classic" <?php selected($editor_choice, 'classic'); ?>><?php esc_html_e('Classic Editor', 'hi-classic'); ?></option>
        <option value="gutenberg" <?php selected($editor_choice, 'gutenberg'); ?>><?php esc_html_e('Gutenberg Editor', 'hi-classic'); ?></option>
    </select>
    <p class="description"><?php esc_html_e('Select the default editor for your posts and pages.', 'hi-classic'); ?></p>
    <?php
}

// Set the editor based on user preference
function mytheme_set_editor() {
    $editor_choice = get_option('mytheme_editor_choice', 'classic');

    if ($editor_choice === 'classic') {
        add_filter('use_block_editor_for_post', '__return_false', 10);
    } else {
        add_filter('use_block_editor_for_post', '__return_true', 10);
    }
}
add_action('admin_init', 'mytheme_set_editor');

// Display the custom header content
function mytheme_display_header() {
    $header_template_id = get_option('mytheme_header_template');

    if ($header_template_id && class_exists('\Elementor\Plugin')) {
        echo \Elementor\Plugin::$instance->frontend->get_builder_content_for_display($header_template_id);
    } else {
        ?>
        <header class="site-header">
            <div class="logo">
                <?php 
                if (has_custom_logo()) {
                    echo '<a href="' . home_url() . '" rel="home">';
                    the_custom_logo();
                    echo '</a>';
                } else {
                    echo '<h1><a href="' . home_url() . '" rel="home">' . get_bloginfo('name') . '</a></h1>';
                }
                ?>
            </div>
            <div class="primary--main-menu">
                <?php
                wp_nav_menu(array(
                    'theme_location' => 'primary-menu',
                    'container' => 'nav',
                    'container_class' => 'primary-menu-container',
                    'menu_class' => 'primary-menu',
                ));
                ?>
            </div>
        </header>
        <?php
    }

    $custom_js_code = get_option('mytheme_custom_header_code');
    if (!empty($custom_js_code)) {
        echo '<script type="text/javascript">' . esc_js($custom_js_code) . '</script>';
    }

    $custom_html_code = get_option('mytheme_custom_html_code');
    if (!empty($custom_html_code)) {
        echo $custom_html_code;
    }
}
add_action('wp_head', 'mytheme_display_header');

// Display footer template or default footer
function mytheme_display_footer() {
    $footer_template_id = get_option('mytheme_footer_template');

    if ($footer_template_id && class_exists('\Elementor\Plugin')) {
        echo \Elementor\Plugin::$instance->frontend->get_builder_content_for_display($footer_template_id);
    } else {
        ?>
        <footer class="site-footer">
            <div class="footer-content">
                <p>&copy; <?php echo date('Y'); ?> <?php bloginfo('name'); ?></p>
            </div>
        </footer>
        <?php
    }
}
add_action('wp_footer', 'mytheme_display_footer');
