<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Hi_Classic_Popup_Button extends \Elementor\Widget_Base {

    public function get_name() {
        return 'hi-classic-popup-button';
    }

    public function get_title() {
        return __( 'Hi Classic Popup Button', 'hi-classic-popup' );
    }

    public function get_icon() {
        return 'eicon-button';
    }

    public function get_categories() {
        return [ 'general' ];
    }

    protected function register_controls() {
        // ----------------------
        // Content Section
        // ----------------------
        $this->start_controls_section(
            'content_section',
            [ 'label' => __( 'Content', 'hi-classic-popup' ) ]
        );

        $this->add_control(
            'text',
            [
                'label' => __( 'Text', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __( 'Open Popup', 'hi-classic-popup' ),
                'placeholder' => __( 'Enter button text', 'hi-classic-popup' ),
            ]
        );

        $this->add_control(
            'template_id',
            [
                'label' => __( 'Choose Popup Template', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => $this->get_elementor_templates(),
                'default' => '',
            ]
        );

        $this->add_control(
            'icon',
            [
                'label' => __( 'Icon', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::ICONS,
            ]
        );

        $this->add_control(
            'icon_align',
            [
                'label' => __( 'Icon Position', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'left',
                'options' => [
                    'left'  => __( 'Before', 'hi-classic-popup' ),
                    'right' => __( 'After', 'hi-classic-popup' ),
                ],
                'condition' => [ 'icon[value]!' => '' ],
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => __( 'Alignment', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left'   => [ 'title' => __( 'Left', 'hi-classic-popup' ), 'icon' => 'eicon-text-align-left' ],
                    'center' => [ 'title' => __( 'Center', 'hi-classic-popup' ), 'icon' => 'eicon-text-align-center' ],
                    'right'  => [ 'title' => __( 'Right', 'hi-classic-popup' ), 'icon' => 'eicon-text-align-right' ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .hi-popup-btn-wrapper' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // ----------------------
        // Button Style Section
        // ----------------------
        $this->start_controls_section(
            'style_section',
            [ 'label' => __( 'Button', 'hi-classic-popup' ), 'tab' => \Elementor\Controls_Manager::TAB_STYLE ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => __( 'Text Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [ '{{WRAPPER}} .hi-popup-btn' => 'color: {{VALUE}};' ],
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __( 'Background Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [ '{{WRAPPER}} .hi-popup-btn' => 'background-color: {{VALUE}};' ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'selector' => '{{WRAPPER}} .hi-popup-btn',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .hi-popup-btn',
            ]
        );

        $this->add_control(
            'border_radius',
            [
                'label' => __( 'Border Radius', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'selectors' => [ '{{WRAPPER}} .hi-popup-btn' => 'border-radius: {{SIZE}}{{UNIT}};' ],
            ]
        );

        $this->add_control(
            'hover_color',
            [
                'label' => __( 'Hover Text Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [ '{{WRAPPER}} .hi-popup-btn:hover' => 'color: {{VALUE}};' ],
            ]
        );

        $this->add_control(
            'hover_background',
            [
                'label' => __( 'Hover Background', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [ '{{WRAPPER}} .hi-popup-btn:hover' => 'background-color: {{VALUE}};' ],
            ]
        );

        $this->end_controls_section();

        // ----------------------
        // Close Button Style
        // ----------------------
        $this->start_controls_section(
            'close_button_style_section',
            [ 'label' => __( 'Close Button', 'hi-classic-popup' ), 'tab' => \Elementor\Controls_Manager::TAB_STYLE ]
        );

        $this->add_control(
            'close_button_color',
            [
                'label' => __( 'Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [ 'body .hi-popup-close' => 'color: {{VALUE}};' ],
            ]
        );

        $this->add_control(
            'close_button_size',
            [
                'label' => __( 'Size', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 10, 'max' => 100 ] ],
                'selectors' => [ 'body .hi-popup-close' => 'font-size: {{SIZE}}{{UNIT}};' ],
            ]
        );

        $this->add_responsive_control(
            'close_button_position_top',
            [
                'label' => __( 'Top Position', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 0, 'max' => 200 ] ],
                'selectors' => [ 'body .hi-popup-close' => 'top: {{SIZE}}{{UNIT}};' ],
            ]
        );

        $this->add_responsive_control(
            'close_button_position_right',
            [
                'label' => __( 'Right Position', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 0, 'max' => 200 ] ],
                'selectors' => [ 'body .hi-popup-close' => 'right: {{SIZE}}{{UNIT}};' ],
            ]
        );

        $this->add_responsive_control(
            'close_button_padding',
            [
                'label' => __( 'Padding', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    'body .hi-popup-close' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // ----------------------
        // Icon Style
        // ----------------------
        $this->start_controls_section(
            'icon_style_section',
            [
                'label' => __( 'Icon', 'hi-classic-popup' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [ 'icon[value]!' => '' ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __( 'Icon Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .hi-popup-btn i, {{WRAPPER}} .hi-popup-btn svg' => 'color: {{VALUE}}; fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_hover_color',
            [
                'label' => __( 'Hover Icon Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .hi-popup-btn:hover i, {{WRAPPER}} .hi-popup-btn:hover svg' => 'color: {{VALUE}}; fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __( 'Icon Size', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 10, 'max' => 100 ] ],
                'selectors' => [
                    '{{WRAPPER}} .hi-popup-btn i, {{WRAPPER}} .hi-popup-btn svg' => 'font-size: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label' => __( 'Icon Spacing', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 0, 'max' => 50 ] ],
                'selectors' => [
                    '{{WRAPPER}} .hi-popup-btn .elementor-icon-left.hi-popup-icon'  => 'margin-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .hi-popup-btn .elementor-icon-right.hi-popup-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // ----------------------
        // Popup Content Style
        // ----------------------
        $this->start_controls_section(
            'popup_content_style_section',
            [
                'label' => __( 'Popup Content', 'hi-classic-popup' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'popup_background_color',
            [
                'label' => __( 'Background Color', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    'body .hi-popup-content' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'popup_padding',
            [
                'label' => __( 'Padding', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'selectors' => [
                    'body .hi-popup-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'popup_border_radius',
            [
                'label' => __( 'Border Radius', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'selectors' => [
                    'body .hi-popup-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'popup_max_width',
            [
                'label' => __( 'Max Width', 'hi-classic-popup' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 100, 'max' => 2000 ] ],
                'selectors' => [
                    'body .hi-popup-content' => 'max-width: {{SIZE}}{{UNIT}}; margin: 0 auto;',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'popup_box_shadow',
                'selector' => 'body .hi-popup-content',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Fetch only “Hi Classic Popup” templates.
     */
    private function get_elementor_templates() {
        $templates = get_transient( 'hi_classic_popup_templates_dropdown' );
        if ( $templates !== false ) {
            return $templates;
        }

        $args = [
            'post_type'      => 'elementor_library',
            'post_status' => ['publish'],
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
            'fields'         => 'ids',
        ];

        $template_ids = get_posts( $args );
        $templates    = [];

        foreach ( $template_ids as $id ) {
            $type = get_post_meta( $id, '_elementor_template_type', true );
            if ( $type === 'hi-classic-popup' ) {
                $templates[ $id ] = esc_html( get_the_title( $id ) );
            }
        }

        set_transient( 'hi_classic_popup_templates_dropdown', $templates, HOUR_IN_SECONDS );
        return $templates;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute( 'button', 'class', 'hi-popup-btn' );
        $this->add_render_attribute( 'button', 'data-template-id', esc_attr( $settings['template_id'] ) );

        echo '<div class="hi-popup-btn-wrapper">';
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Safe Elementor attribute rendering.
        echo '<button ' . $this->get_render_attribute_string( 'button' ) . '>';

        if ( ! empty( $settings['icon']['value'] ) && 'left' === $settings['icon_align'] ) {
            echo '<span class="hi-popup-icon elementor-icon-left">';
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Safe Elementor icon rendering.
            \Elementor\Icons_Manager::render_icon( $settings['icon'], [ 'aria-hidden' => 'true' ] );
            echo '</span>';
        }

        echo '<span class="hi-popup-btn-text">' . esc_html( $settings['text'] ) . '</span>';

        if ( ! empty( $settings['icon']['value'] ) && 'right' === $settings['icon_align'] ) {
            echo '<span class="hi-popup-icon elementor-icon-right">';
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Safe Elementor icon rendering.
            \Elementor\Icons_Manager::render_icon( $settings['icon'], [ 'aria-hidden' => 'true' ] );
            echo '</span>';
        }

        echo '</button>';
        echo '</div>';
    }
}

// Auto-clear cache when Elementor templates are saved or deleted
add_action( 'save_post_elementor_library', function() {
    delete_transient( 'hi_classic_popup_templates_dropdown' );
});

add_action( 'delete_post', function( $post_id ) {
    if ( get_post_type( $post_id ) === 'elementor_library' ) {
        delete_transient( 'hi_classic_popup_templates_dropdown' );
    }
});
