<?php
/**
 * Plugin Name: Hi Classic Form
 * Plugin URI:  https://hiclassic.site/hi-classic-form
 * Description: Elementor-compatible contact form widget with AJAX submission, per-field columns, style controls, labels toggle, and admin test email.
 * Version: 1.4.0
 * Author: Hi Classic
 * Author URI: https://hiclassic.site/
 * Text Domain: hi-classic-form
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'HI_CLASSIC_FORM_PATH', plugin_dir_path( __FILE__ ) );
define( 'HI_CLASSIC_FORM_URL', plugin_dir_url( __FILE__ ) );
define( 'HI_CLASSIC_FORM_VERSION', '1.4.1' );

/**
 * Register Elementor widget
 */
add_action( 'plugins_loaded', function() {
    if ( ! did_action( 'elementor/loaded' ) ) {
        add_action( 'admin_notices', function() {
            if ( current_user_can( 'activate_plugins' ) ) {
                echo '<div class="notice notice-warning"><p>Hi Classic Form requires <strong>Elementor</strong> to be installed and active.</p></div>';
            }
        } );
        return;
    }

    add_action( 'elementor/widgets/register', function( $widgets_manager ) {
        require_once HI_CLASSIC_FORM_PATH . 'hi-classic-form-widget.php';
        $widgets_manager->register( new \Hi_Classic_Form_Widget() );
    } );
} );

/**
 * Enqueue frontend assets
 */
add_action( 'wp_enqueue_scripts', function() {
    wp_register_script(
        'hi-classic-form-frontend',
        HI_CLASSIC_FORM_URL . 'assets/js/hi-classic-form.js',
        array( 'jquery' ),
        HI_CLASSIC_FORM_VERSION,
        true
    );

    // Localize script for AJAX + reCAPTCHA site key
    $hi_recaptcha_site = get_option( 'hi_classic_form_recaptcha_site_key', '' );
    wp_localize_script( 'hi-classic-form-frontend', 'HiClassicForm', array(
        'ajax_url'           => admin_url( 'admin-ajax.php' ),
        'nonce'              => wp_create_nonce( 'hi_classic_form' ),
        'recaptcha_site_key' => $hi_recaptcha_site,
    ) );

    wp_enqueue_script( 'hi-classic-form-frontend' );

    wp_register_style(
        'hi-classic-form-css',
        HI_CLASSIC_FORM_URL . 'assets/css/hi-classic-form.css',
        array(),
        HI_CLASSIC_FORM_VERSION
    );
    wp_enqueue_style( 'hi-classic-form-css' );

    // Conditionally enqueue Google reCAPTCHA v3 script
    if ( ! empty( $hi_recaptcha_site ) ) {
        wp_enqueue_script(
            'hi-classic-form-recaptcha',
            'https://www.google.com/recaptcha/api.js?render=' . esc_attr( $hi_recaptcha_site ),
            array(),
            HI_CLASSIC_FORM_VERSION,
            true
        );
    }
} );

/**
 * Handle AJAX form submission
 */
add_action( 'wp_ajax_hi_classic_form_submit', 'hi_classic_form_handle_submission' );
add_action( 'wp_ajax_nopriv_hi_classic_form_submit', 'hi_classic_form_handle_submission' );

function hi_classic_form_handle_submission() {
    $response = array( 'success' => false );

    // Nonce verification
    if ( empty( $_POST['_hi_classic_form_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_hi_classic_form_nonce'] ) ), 'hi_classic_form' ) ) {
        $response['message'] = __( 'Security check failed.', 'hi-classic-form' );
        wp_send_json( $response );
    }

    /**
     * reCAPTCHA v3 verification
     */
    $hi_recaptcha_secret = get_option( 'hi_classic_form_recaptcha_secret_key', '' );
    if ( ! empty( $hi_recaptcha_secret ) ) {
        $token = '';
        if ( isset( $_POST['g-recaptcha-response'] ) ) {
            $token = sanitize_text_field( wp_unslash( $_POST['g-recaptcha-response'] ) );
        } elseif ( isset( $_POST['form']['g-recaptcha-response'] ) ) {
            $token = sanitize_text_field( wp_unslash( $_POST['form']['g-recaptcha-response'] ) );
        }

        if ( empty( $token ) ) {
            $response['message'] = __( 'reCAPTCHA verification failed. Please try again.', 'hi-classic-form' );
            wp_send_json( $response );
        }

        $remoteip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';

        // Debug reCAPTCHA
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( '=== Hi Classic Form reCAPTCHA Verification ===' );
            error_log( 'Secret key: ' . substr( $hi_recaptcha_secret, 0, 8 ) . '********' );
        }

        $verify = wp_remote_post( 'https://www.google.com/recaptcha/api/siteverify', array(
            'body' => array(
                'secret'   => $hi_recaptcha_secret,
                'response' => $token,
                'remoteip' => $remoteip,
            ),
            'timeout' => 15,
        ) );

        if ( is_wp_error( $verify ) ) {
            $response['message'] = __( 'reCAPTCHA verification failed. Please try again later.', 'hi-classic-form' );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'reCAPTCHA WP_Error: ' . $verify->get_error_message() );
            }
            wp_send_json( $response );
        }

        $body   = wp_remote_retrieve_body( $verify );
        $result = json_decode( $body, true );

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'reCAPTCHA response: ' . print_r( $result, true ) );
        }

        $score_ok = true;
        if ( isset( $result['score'] ) ) {
            $score_ok = (float) $result['score'] >= 0.1;
            error_log( 'reCAPTCHA score: ' . $result['score'] );
        }

        if ( empty( $result ) || empty( $result['success'] ) || ! $score_ok ) {
            $response['message'] = __( 'reCAPTCHA verification failed. Please try again.', 'hi-classic-form' );
            wp_send_json( $response );
        }
    }

    /**
     * Gather form fields
     */
    $raw_form = array();
    if ( isset( $_POST['form'] ) && is_array( $_POST['form'] ) ) {
        $raw_form = map_deep( wp_unslash( $_POST['form'] ), 'sanitize_text_field' );
    }

    if ( ! is_array( $raw_form ) ) {
        $response['message'] = __( 'Invalid form submission.', 'hi-classic-form' );
        wp_send_json( $response );
    }

    $fields = array();
    foreach ( $raw_form as $k => $v ) {
        $fields[ sanitize_text_field( $k ) ] = wp_kses_post( $v );
    }

    /**
     * Email setup
     */
    $to = isset( $_POST['to'] ) ? sanitize_email( wp_unslash( $_POST['to'] ) ) : get_option( 'admin_email' );
    if ( empty( $to ) ) {
        $to = get_option( 'admin_email' );
    }

		if ( ! empty( $fields['subject'] ) ) {
			$subject = sanitize_text_field( $fields['subject'] );
		} else {
			/* translators: %s: site name (blog name). */
			$subject = sprintf( __( 'New message from %s', 'hi-classic-form' ), get_bloginfo( 'name' ) );
		}


    $message_lines = array();
    foreach ( $fields as $label => $val ) {
        $message_lines[] = sprintf( '%s: %s', $label, wp_strip_all_tags( $val ) );
    }
    $message = implode( "\n", $message_lines );

    $headers = array();
    $headers[] = 'Content-Type: text/plain; charset=' . get_option( 'blog_charset' );
    $site_domain = str_replace( array( 'https://', 'http://' ), '', home_url() );
    $site_name = get_bloginfo( 'name' );
    $headers[] = 'From: ' . $site_name . ' <no-reply@' . $site_domain . '>';

    $possible_email_keys = array( 'email', 'Email', 'e-mail', 'your_email' );
    foreach ( $possible_email_keys as $k ) {
        if ( ! empty( $fields[ $k ] ) && is_email( $fields[ $k ] ) ) {
            $headers[] = 'Reply-To: ' . sanitize_email( $fields[ $k ] );
            break;
        }
    }

    /**
     * Send email + debug
     */
    $sent = wp_mail( $to, $subject, $message, $headers );

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        global $phpmailer;
        error_log( '=== Hi Classic Form Email Debug ===' );
        error_log( 'To: ' . $to );
        error_log( 'Subject: ' . $subject );
        error_log( 'Headers: ' . print_r( $headers, true ) );
        error_log( 'Message: ' . $message );
        error_log( 'Result: ' . ( $sent ? 'SUCCESS ✅' : 'FAILED ❌' ) );
        if ( isset( $phpmailer ) && $phpmailer instanceof PHPMailer ) {
            error_log( 'PHPMailer Error Info: ' . $phpmailer->ErrorInfo );
        }
    }

    /**
     * Return response
     */
    if ( $sent ) {
        $response['success'] = true;
        $response['message'] = isset( $_POST['success_message'] )
            ? sanitize_text_field( wp_unslash( $_POST['success_message'] ) )
            : __( 'Thanks for your message. We will get back to you soon.', 'hi-classic-form' );
    } else {
        $response['message'] = __( 'Failed to send email. Please check your mail configuration (SMTP).', 'hi-classic-form' );
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            $response['debug'] = 'Check debug.log for PHPMailer error details.';
        }
    }

    wp_send_json( $response );
}

/**
 * Include admin pages
 */
require_once __DIR__ . '/admin/test-email-page.php';
require_once __DIR__ . '/admin/recaptcha-settings.php';
