<?php
if ( ! defined( 'ABSPATH' ) ) exit;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Typography;

class Hi_Classic_Carousel_Widget extends Widget_Base {

    public function get_name() { return 'hi_classic_carousel'; }
    public function get_title() { return esc_html__( 'Hi Classic Carousel', 'hi-classic-carousel' ); }
    public function get_icon() { return 'eicon-slider-push'; }
    public function get_categories() { return array( 'general', 'basic' ); }
    public function get_style_depends() { return array('hi-classic-carousel-style'); }

    protected function get_elementor_templates() {
        $cached = get_transient('hi_classic_carousel_templates_dropdown');
        if ($cached !== false) {
            return $cached;
        }

        $options = array();
        if ( class_exists( '\Elementor\Plugin' ) ) {
            try {
                $source = \Elementor\Plugin::instance()->templates_manager->get_source('local');
                if ( $source ) {
                    $items = $source->get_items();
                    if ( is_array( $items ) ) {
                        foreach ( $items as $item ) {
                            $id = $item['template_id'] ?? $item['post_id'] ?? null;
                            $title = $item['title'] ?? $item['post_title'] ?? '';
                            if ( $id ) {
                                $t_type = get_post_meta( intval($id), '_elementor_template_type', true );
                                if ( $t_type === 'hi-classic-carousel' ) {
                                    $options[ $id ] = $title;
                                }
                            }
                        }
                    }
                }
            } catch ( \Exception $e ) {}
        }

        set_transient('hi_classic_carousel_templates_dropdown', $options, HOUR_IN_SECONDS);
        return $options;
    }

    protected function register_controls() {

        $this->start_controls_section('content_section',[
            'label'=>esc_html__('Slides','hi-classic-carousel'),
            'tab'=>Controls_Manager::TAB_CONTENT,
        ]);

        $repeater = new Repeater();

        $repeater->add_control('slide_type',[
            'label'=>esc_html__('Slide Type','hi-classic-carousel'),
            'type'=>Controls_Manager::SELECT,
            'options'=>[ 'custom'=>'Custom','template'=>'Elementor Template' ],
            'default'=>'custom',
        ]);

        $repeater->add_control('slide_image',['label'=>esc_html__('Image','hi-classic-carousel'),'type'=>Controls_Manager::MEDIA]);
        $repeater->add_control('slide_title',['label'=>esc_html__('Title','hi-classic-carousel'),'type'=>Controls_Manager::TEXT]);
        $repeater->add_control('slide_caption',['label'=>esc_html__('Caption','hi-classic-carousel'),'type'=>Controls_Manager::TEXTAREA]);
        $repeater->add_control('slide_template',['label'=>esc_html__('Choose Template','hi-classic-carousel'),'type'=>Controls_Manager::SELECT,'options'=>$this->get_elementor_templates()]);

        $this->add_control('slides',[
            'label'=>esc_html__('Slides','hi-classic-carousel'),
            'type'=>Controls_Manager::REPEATER,
            'fields'=>$repeater->get_controls(),
            'title_field'=>'{{{ slide_title }}}',
        ]);

        // ✅ Responsive slides per view
        $this->add_responsive_control('slides_per_view', [
            'label' => esc_html__('Slides Per View','hi-classic-carousel'),
            'type' => Controls_Manager::NUMBER,
            'devices' => [ 'desktop', 'tablet', 'mobile' ],
            'desktop_default' => 3,
            'tablet_default' => 2,
            'mobile_default' => 1,
            'min' => 1,
            'max' => 6,
        ]);

        $this->add_control('autoplay_enabled', [
            'label' => __('Autoplay','hi-classic-carousel'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('Yes','hi-classic-carousel'),
            'label_off' => __('No','hi-classic-carousel'),
            'return_value' => 'yes',
            'default' => 'yes',
        ]);

        $this->add_control('autoplay_interval', [
            'label' => __('Autoplay Interval (ms)','hi-classic-carousel'),
            'type' => Controls_Manager::NUMBER,
            'default' => 5000,
            'condition' => [ 'autoplay_enabled' => 'yes' ],
        ]);

        // ✅ New Infinite Loop toggle
        $this->add_control('infinite_loop', [
            'label' => __('Infinite Loop','hi-classic-carousel'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('Yes','hi-classic-carousel'),
            'label_off' => __('No','hi-classic-carousel'),
            'return_value' => 'yes',
            'default' => 'no',
        ]);

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section('style_section',[
            'label'=>esc_html__('Slide Style','hi-classic-carousel'),
            'tab'=>Controls_Manager::TAB_STYLE,
        ]);

        $this->add_control('title_color',['label'=>__('Title Color','hi-classic-carousel'),'type'=>Controls_Manager::COLOR,'selectors'=>['{{WRAPPER}} .carousel-caption h5'=>'color: {{VALUE}};']]);
        $this->add_group_control(Group_Control_Typography::get_type(),['name'=>'title_typography','selector'=>'{{WRAPPER}} .carousel-caption h5']);
        $this->add_control('caption_color',['label'=>__('Caption Color','hi-classic-carousel'),'type'=>Controls_Manager::COLOR,'selectors'=>['{{WRAPPER}} .carousel-caption p'=>'color: {{VALUE}};']]);
        $this->add_group_control(Group_Control_Typography::get_type(),['name'=>'caption_typography','selector'=>'{{WRAPPER}} .carousel-caption p']);

        $this->add_control('show_dots',[
            'label'=>__('Show Pagination Dots','hi-classic-carousel'),
            'type'=>Controls_Manager::SWITCHER,
            'label_on'=>__('Show','hi-classic-carousel'),
            'label_off'=>__('Hide','hi-classic-carousel'),
            'return_value'=>'yes',
            'default'=>'yes'
        ]);
        $this->add_control('dots_color',['label'=>__('Dot Color','hi-classic-carousel'),'type'=>Controls_Manager::COLOR,'selectors'=>['{{WRAPPER}} .carousel-indicators button'=>'background-color: {{VALUE}};']]);
        $this->add_control('dots_active_color',['label'=>__('Active Dot Color','hi-classic-carousel'),'type'=>Controls_Manager::COLOR,'selectors'=>['{{WRAPPER}} .carousel-indicators .active'=>'background-color: {{VALUE}};']]);
        $this->add_control('dots_size',['label'=>__('Dot Size','hi-classic-carousel'),'type'=>Controls_Manager::SLIDER,'size_units'=>['px','em','rem'],'range'=>['px'=>['min'=>4,'max'=>40,'step'=>1]],'selectors'=>['{{WRAPPER}} .carousel-indicators button'=> 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}; border-radius:50%;']]);

        $this->add_control('show_arrows',[
            'label'=>__('Show Navigation Arrows','hi-classic-carousel'),
            'type'=>Controls_Manager::SWITCHER,
            'label_on'=>__('Show','hi-classic-carousel'),
            'label_off'=>__('Hide','hi-classic-carousel'),
            'return_value'=>'yes',
            'default'=>'yes'
        ]);
        $this->add_control('arrow_color',['label'=>__('Arrow Color','hi-classic-carousel'),'type'=>Controls_Manager::COLOR,'selectors'=>['{{WRAPPER}} .carousel-control-prev-icon'=> 'background-color: {{VALUE}};','{{WRAPPER}} .carousel-control-next-icon'=> 'background-color: {{VALUE}};']]);
        $this->add_control('arrow_size',['label'=>__('Arrow Size','hi-classic-carousel'),'type'=>Controls_Manager::SLIDER,'size_units'=>['px','em','rem'],'range'=>['px'=>['min'=>10,'max'=>80,'step'=>1]],'selectors'=>['{{WRAPPER}} .carousel-control-prev-icon, {{WRAPPER}} .carousel-control-next-icon'=>'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};']]);

        $this->end_controls_section();
    }

    protected function render() {
        $s = $this->get_settings_for_display();
        $slides = $s['slides'] ?? [];

        // Responsive settings
        $desktop = max(1, intval($s['slides_per_view'] ?? 3));
        $tablet  = max(1, intval($s['slides_per_view_tablet'] ?? 2));
        $mobile  = max(1, intval($s['slides_per_view_mobile'] ?? 1));

        $uid = 'hi-carousel-' . wp_rand(1000, 9999);
        $data_autoplay = (!empty($s['autoplay_enabled']) && $s['autoplay_enabled'] === 'yes') ? intval($s['autoplay_interval']) : 0;
        $data_infinite = (!empty($s['infinite_loop']) && $s['infinite_loop'] === 'yes') ? 'true' : 'false';

echo '<div id="' . esc_attr($uid) . '" 
        class="carousel slide hi-classic-carousel" 
        data-autoplay="' . esc_attr($data_autoplay) . '" 
        data-infinite="true"
        data-bs-ride="' . (!empty($data_autoplay) ? 'carousel' : 'false') . '">';

        echo '<div class="carousel-inner">';

        // Group slides by desktop value
        $chunks = array_chunk($slides, $desktop);
        foreach ($chunks as $i => $group) {
            echo '<div class="carousel-item ' . ($i === 0 ? 'active' : '') . '"><div class="d-flex justify-content-center">';
            foreach ($group as $slide) {
                echo '<div class="slide-item">';

                if (isset($slide['slide_type']) && $slide['slide_type'] === 'template' && !empty($slide['slide_template'])) {
                    $template_id = intval($slide['slide_template']);
                    if (class_exists('\Elementor\Plugin')) {
                        try {
                            $content = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display($template_id);
   if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
	       // Elementor requires raw HTML in edit mode.
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo $content;
} else {
    echo wp_kses_post( $content );
}
                        } catch (\Exception $e) {
                            echo '<div class="hi-carousel-error">' . esc_html__('Unable to load template', 'hi-classic-carousel') . '</div>';
                        }
                    }
                } else {
                    if (!empty($slide['slide_image']['url'])) {
                        echo '<img src="' . esc_url($slide['slide_image']['url']) . '" class="d-block w-100" alt="">';
                    }
                    echo '<div class="hi-slide-caption carousel-caption d-none d-md-block">';
                    if (!empty($slide['slide_title'])) {
                        echo '<h5>' . esc_html($slide['slide_title']) . '</h5>';
                    }
                    if (!empty($slide['slide_caption'])) {
                        echo '<p>' . esc_html($slide['slide_caption']) . '</p>';
                    }
                    echo '</div>';
                }

                echo '</div>'; // .slide-item
            }
            echo '</div></div>'; // flex + carousel-item
        }
        echo '</div>'; // carousel-inner

        // Arrows
        if (!empty($s['show_arrows']) && $s['show_arrows'] === 'yes') {
            echo '<button class="carousel-control-prev" type="button" data-bs-target="#' . esc_attr($uid) . '" data-bs-slide="prev">
                    <span class="carousel-control-prev-icon" aria-hidden="true"></span>
                    <span class="visually-hidden">' . esc_html__('Previous', 'hi-classic-carousel') . '</span>
                  </button>';
            echo '<button class="carousel-control-next" type="button" data-bs-target="#' . esc_attr($uid) . '" data-bs-slide="next">
                    <span class="carousel-control-next-icon" aria-hidden="true"></span>
                    <span class="visually-hidden">' . esc_html__('Next', 'hi-classic-carousel') . '</span>
                  </button>';
        }

        // Dots
        if (!empty($s['show_dots']) && $s['show_dots'] === 'yes') {
            echo '<div class="carousel-indicators">';
            foreach ($chunks as $i => $group) {
                echo '<button type="button" data-bs-target="#' . esc_attr($uid) . '" data-bs-slide-to="' . esc_attr($i) . '" ' . ($i === 0 ? 'class="active" aria-current="true"' : '') . ' aria-label="' . esc_attr__('Slide', 'hi-classic-carousel') . ' ' . esc_attr($i + 1) . '"></button>';
            }
            echo '</div>';
        }

        echo '</div>'; // carousel
echo '<style>
    #' . esc_attr($uid) . ' .slide-item { flex: 0 0 auto; padding:0 5px; }
    @media (min-width: 992px) {
        #' . esc_attr($uid) . ' .slide-item { width: ' . esc_attr( 100 / $desktop ) . '%; }
    }
    @media (min-width: 768px) and (max-width: 991px) {
        #' . esc_attr($uid) . ' .slide-item { width: ' . esc_attr( 100 / $tablet ) . '%; }
    }
    @media (max-width: 767px) {
        #' . esc_attr($uid) . ' .slide-item { width: ' . esc_attr( 100 / $mobile ) . '%; }
    }
</style>';
    }
}
