<?php
/**
 * Plugin Name: Hi-Classic-Carousel
 * Description: Elementor-compatible carousel widget with template and custom slide support.
 * Version: 1.3.0
 * Author: Hi Classic
 * Text Domain: hi-classic-carousel
 * License: GPLv2 or later
 * Author URI:  https://hiclassic.site/
* License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class Hi_Classic_Carousel_Plugin {
    const VERSION = '1.3.0';
    const SLUG = 'hi-classic-carousel';

    private static $instance = null;

    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'frontend_assets' ) );
    }

    public function on_plugins_loaded() {
        if ( did_action( 'elementor/loaded' ) ) {
            add_action( 'elementor/widgets/register', array( $this, 'register_widget' ), 20 );
        } else {
            add_action( 'admin_notices', array( $this, 'elementor_missing_notice' ) );
        }
    }

    public function elementor_missing_notice() {
        if ( current_user_can( 'activate_plugins' ) ) {
            echo '<div class="notice notice-error"><p><strong>Hi-Classic-Carousel</strong> requires <a href="https://wordpress.org/plugins/elementor/" target="_blank">Elementor</a> to be installed and active.</p></div>';
        }
    }

    public function frontend_assets() {
        // Load local Bootstrap CSS & JS
        wp_enqueue_style(
            'bootstrap-css',
            plugins_url( 'assets/css/bootstrap.min.css', __FILE__ ),
            array(),
            '5.3.2'
        );
        wp_enqueue_script(
            'bootstrap-js',
            plugins_url( 'assets/js/bootstrap.bundle.min.js', __FILE__ ),
            array(),
            '5.3.2',
            true
        );

        // Plugin CSS & JS
        wp_enqueue_style(
            'hi-classic-carousel-style',
            plugins_url( 'assets/css/hi-classic-carousel.css', __FILE__ ),
            array( 'bootstrap-css' ),
            self::VERSION
        );

wp_enqueue_script(
    'hi-classic-carousel-init',
    plugins_url('assets/js/hi-classic-carousel.js', __FILE__),
    array('jquery', 'bootstrap-js', 'elementor-frontend'), // important!
    self::VERSION,
    true
);
    }

    public function register_widget( $widgets_manager ) {
        $widget_file = __DIR__ . '/widget-hi-classic-carousel.php';
        if ( file_exists( $widget_file ) ) {
            require_once $widget_file;
            if ( class_exists( 'Hi_Classic_Carousel_Widget' ) ) {
                $widgets_manager->register( new Hi_Classic_Carousel_Widget() );
            }
        }
    }
}

/**
 * Register new Elementor template type "Hi Classic Carousel"
 *
 * This adds the "Hi Classic Carousel" option to Templates -> Add New dropdown
 */
add_filter( 'elementor_library_get_template_types', function( $types ) {
    $types['hi-classic-carousel'] = [
        'title' => __( 'Hi Classic Carousel', 'hi-classic-carousel' ),
        'show_in_ui' => true,
    ];
    return $types;
} );

/**
 * Register custom Elementor document for hi-classic-carousel (safe timing)
 */
add_action( 'elementor/documents/register', function( $documents_manager ) {
    $file = __DIR__ . '/includes/class-hi-classic-carousel-document.php';
    if ( file_exists( $file ) ) {
        require_once $file;
        if ( class_exists( 'Hi_Classic_Carousel_Document' ) ) {
            $documents_manager->register_document_type( 'hi-classic-carousel', 'Hi_Classic_Carousel_Document' );
        }
    }
}, 11 );

// ✅ Clear cached template dropdown on save or delete
add_action( 'save_post_elementor_library', function( $post_id ) {
    $type = get_post_meta( $post_id, '_elementor_template_type', true );
    if ( $type === 'hi-classic-carousel' ) {
        delete_transient( 'hi_classic_carousel_templates_dropdown' );
    }
}); 

add_action( 'delete_post_elementor_library', function( $post_id ) {
    $type = get_post_meta( $post_id, '_elementor_template_type', true );
    if ( $type === 'hi-classic-carousel' ) {
        delete_transient( 'hi_classic_carousel_templates_dropdown' );
    }
});

// Load only in Elementor editor
add_action( 'elementor/editor/after_enqueue_scripts', function() {
    wp_enqueue_script(
        'hi-classic-carousel-refresh',
        plugins_url( 'assets/js/hi-carousel-refresh.js', __FILE__ ),
        array( 'jquery', 'elementor-editor' ),
        Hi_Classic_Carousel_Plugin::VERSION,
        true
    );
});
// Initialize the plugin
Hi_Classic_Carousel_Plugin::instance();
